<?php
namespace AIOSEO\Plugin\Addon\LinkAssistant\Api;

use AIOSEO\Plugin\Addon\LinkAssistant\Models;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Handles all endpoints for the inner Domains Report.
 *
 * @since 1.0.0
 */
class DomainsReportInner {
	/**
	 * The domain posts limit.
	 *
	 * @since 1.0.0
	 *
	 * @var int
	 */
	private static $domainPostsLimit = 5;

	/**
	 * Returns paginated domain posts results.
	 *
	 * @since 1.0.0
	 *
	 * @param  \WP_Request  $request The request.
	 * @return \WP_Response          The response.
	 */
	public static function paginate( $request ) {
		$page        = (int) $request['page'];
		$hostname    = sanitize_text_field( $request['hostname'] );
		$offset      = 1 === $page ? 0 : ( $page - 1 ) * self::$domainPostsLimit;
		$domainPosts = Models\Link::getDomainPostLinks( $hostname, $offset );

		if ( empty( $domainPosts ) ) {
			return new \WP_REST_Response( [
				'success' => false,
				'error'   => 'No rows were found.'
			], 404 );
		}

		return new \WP_REST_Response( [
			'success' => true,
			'posts'   => $domainPosts
		], 200 );
	}

	/**
	 * Processes a bulk action on posts linking to a domain.
	 *
	 * @since 1.0.0
	 *
	 * @param  \WP_Request  $request The request.
	 * @return \WP_Response          The response.
	 */
	public static function bulk( $request ) {
		$body        = $request->get_json_params();
		$currentPage = ! empty( $body['currentPage'] ) ? (int) $body['currentPage'] : 1;
		$offset      = 1 === $currentPage ? 0 : ( $currentPage - 1 ) * self::$domainPostsLimit;
		$searchTerm  = ! empty( $body['searchTerm'] ) ? $body['searchTerm'] : '';
		$links       = ! empty( $body['links'] ) ? $body['links'] : [];

		if ( empty( $links ) ) {
			return new \WP_REST_Response( [
				'success' => false,
				'error'   => 'No valid links were passed.'
			], 400 );
		}

		aioseoLinkAssistant()->helpers->deleteLinksInPost( $links );

		return new \WP_REST_Response( [
			'success'       => true,
			'domainsReport' => aioseoLinkAssistant()->helpers->getDomainsReportData( $offset, $searchTerm )
		], 200 );
	}

	/**
	 * Deletes a given link.
	 *
	 * @since 1.0.0
	 *
	 * @param  \WP_Request  $request The request.
	 * @return \WP_Response          The response.
	 */
	public static function deleteLink( $request ) {
		$body        = $request->get_json_params();
		$currentPage = ! empty( $body['currentPage'] ) ? (int) $body['currentPage'] : 1;
		$offset      = 1 === $currentPage ? 0 : ( $currentPage - 1 ) * self::$domainPostsLimit;
		$searchTerm  = ! empty( $body['searchTerm'] ) ? $body['searchTerm'] : '';
		$link        = ! empty( $body['link'] ) ? $body['link'] : [];

		if ( empty( $link ) ) {
			return new \WP_REST_Response( [
				'success' => false,
				'error'   => 'No valid link was passed.'
			], 400 );
		}

		aioseoLinkAssistant()->helpers->deleteLinksInPost( $link['id'] );

		return new \WP_REST_Response( [
			'success'       => true,
			'domainsReport' => aioseoLinkAssistant()->helpers->getDomainsReportData( $offset, $searchTerm )
		], 200 );
	}

	/**
	 * Updates a given link.
	 *
	 * @since 1.0.0
	 *
	 * @param  \WP_Request  $request The request.
	 * @return \WP_Response          The response.
	 */
	public static function updateLink( $request ) {
		$body        = $request->get_json_params();
		$currentPage = ! empty( $body['currentPage'] ) ? (int) $body['currentPage'] : 1;
		$offset      = 1 === $currentPage ? 0 : ( $currentPage - 1 ) * self::$domainPostsLimit;
		$hostname    = ! empty( $body['hostname'] ) ? $body['hostname'] : [];
		$newLink     = ! empty( $body['link'] ) ? $body['link'] : [];

		if ( empty( $newLink ) || empty( $hostname ) ) {
			return new \WP_REST_Response( [
				'success' => false,
				'error'   => 'No valid link or hostname was passed.'
			], 400 );
		}

		$success = self::updateLinkInPost( $newLink['post_id'], $newLink['id'], $newLink['phrase_html'] );
		if ( ! $success ) {
			return new \WP_REST_Response( [
				'success' => false,
				'error'   => 'Couldn\'t update the post.'
			], 400 );
		}

		$domainPosts = Models\Link::getDomainPostLinks( $hostname, $offset );

		return new \WP_REST_Response( [
			'success' => true,
			'posts'   => $domainPosts
		], 200 );
	}

	/**
	 * Adds a link to the given post or updates an existing one.
	 *
	 * @since 1.0.0
	 *
	 * @param  int    $postId        The post ID.
	 * @param  int    $linkId        The Link ID.
	 * @param  string $newPhraseHtml The HTML of the updated phrase, which includes the Link.
	 * @return bool                  Whether the link was successfully updated.
	 */
	private static function updateLinkInPost( $postId, $linkId, $newPhraseHtml ) {
		// Get the Link object first. We need the original phrase so that we can replace it in the post content.
		$link = Models\Link::getLinkById( $linkId );
		if ( ! $link->exists() ) {
			return false;
		}

		$post = aioseo()->helpers->getPost( $postId );
		if ( ! is_object( $post ) ) {
			return false;
		}

		// Replace the old HTML phrase with the new one.
		$postContent   = preg_replace( '/&nbsp;/', ' ', $post->post_content );
		$newPhraseHtml = aioseoLinkAssistant()->helpers->wpKsesPhrase( $newPhraseHtml );
		if ( ! $newPhraseHtml ) {
			return false;
		}

		$oldPhrase = aioseo()->helpers->escapeRegex( $link->phrase_html );
		$pattern   = "/$oldPhrase/i";

		$postContent = preg_replace( $pattern, $newPhraseHtml, $postContent );

		// Confirm that the old phrase is no longer there.
		if ( preg_match( $pattern, $postContent ) ) {
			return false;
		}

		// Finally, we update the post with the modified post content.
		// We don't need to manually update the Link record because the "save_post" action will automatically re-scan the post.
		$error = wp_update_post( [
			'ID'           => $postId,
			'post_content' => $postContent
		], true );

		if ( 0 === $error || is_a( $error, 'WP_Error' ) ) {
			return false;
		}

		return true;
	}
}