<?php
namespace AIOSEO\Plugin\Addon\LinkAssistant\Api;

use AIOSEO\Plugin\Addon\LinkAssistant\Models;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Handles all endpoints for the Post Report.
 *
 * @since 1.0.0
 */
class PostReport extends Common {
	/**
	 * The links limit.
	 *
	 * @since 1.0.0
	 *
	 * @var int
	 */
	protected static $linksLimit = 10;

	/**
	 * Returns the initial data for the post report.
	 *
	 * @since 1.0.0
	 *
	 * @param  \WP_REST_Request  $request The REST Request
	 * @return \WP_REST_Response          The response.
	 */
	public static function initialData( $request ) {
		$postId = (int) $request['postId'];
		if ( ! $postId ) {
			return new \WP_REST_Response( [
				'success' => false,
				'error'   => 'No valid post ID was passed.'
			], 200 );
		}

		$post = aioseo()->helpers->getPost( $postId );

		return new \WP_REST_Response( [
			'success' => true,
			'links'   => aioseoLinkAssistant()->helpers->getPostLinks( $postId, 0, self::$linksLimit ),
			'context' => [
				'postTitle'   => aioseo()->helpers->decodeHtmlEntities( $post->post_title ),
				'publishDate' => $post->post_date,
				'permalink'   => get_permalink( $postId ),
				'editLink'    => get_edit_post_link( $postId, '' )
			]
		], 200 );
	}

	/**
	 * Returns paginated link results for the Post Report.
	 *
	 * @since 1.0.0
	 *
	 * @param  \WP_Request  $request The request.
	 * @return \WP_Response          The response.
	 */
	public static function paginate( $request ) {
		$type   = ! empty( $request['type'] ) ? $request['type'] : '';
		$postId = ! empty( $request['postId'] ) ? (int) $request['postId'] : 0;

		$page   = (int) $request['page'];
		$offset = 1 === $page ? 0 : ( $page - 1 ) * self::$linksLimit;

		if ( empty( $page ) || empty( $postId ) ) {
			return new \WP_REST_Response( [
				'success' => false,
				'error'   => 'No valid post ID or page number were passed.'
			], 404 );
		}

		return new \WP_REST_Response( [
			'success' => true,
			'links'   => aioseoLinkAssistant()->helpers->getPostLinks( $postId, $offset, self::$linksLimit, $type )
		], 200 );
	}
}